using System;
using System.Diagnostics;
using gov.va.med.vbecs.Common.Log;
using VBECS.Services.Common.AppServices;
using gov.va.med.vbecs.DAL.HL7.OpenLibrary.Messages;
using gov.va.med.vbecs.DAL.HL7.Parsers;
using gov.va.med.vbecs.DAL.HL7AL;

namespace gov.va.med.VBECS.HL7Server.Core
{
	#region Header

		//<Package>Package: VBECS - VistA Blood Establishment Computer System</Package>
		//<Warning> WARNING: Per VHA Directive $VADIRECTIVE this class should not be modified</Warning>
		//<MedicalDevice> Medical Device #: $MEDDEVICENO</MedicalDevice>
		//<Developers>
		//	<Developer>Brian Tomlin</Developer>
		//</Developers>
		//<SiteName>Hines OIFO</SiteName>
		//<CreationDate>5/10/2004</CreationDate>
		//<Note>The Food and Drug Administration classifies this software as a medical device.  As such, it may not be changed in any way. Modifications to this software may result in an adulterated medical device under 21CFR820, the use of which is considered to be a violation of US Federal Statutes.  Acquiring and implementing this software through the Freedom of information Act requires the implementor to assume total responsibility for the software, and become a registered manufacturer of a medical device, subject to FDA regulations</Note>
		//<summary>The SimpleListenerHL7Manager is used by the SimpleListener to route incoming HL7 messages to the appropriate interface parser and to return the appropriate response to the incoming HL7 message when a response is required.</summary>

		#endregion

	/// <summary>
	/// Class SimpleListenerHL7Manager
	/// </summary>
	public class SimpleListenerHL7Manager
	{
		// CR 2962
		private const string BCE_COTS_INTERFACE_NAME = "BCE_COTS";
		private const string CPRS_INTERFACE_NAME = "CPRS";
		private const string PATIENT_UPDATE_INTERFACE_NAME = "Patient_Update";
		private const string PATIENT_MERGE_INTERFACE_NAME = "Patient_Merge";
        private const string AUTO_INSTRUMENT_INTERFACE_NAME = "Auto_Instrument";
        // Events Logger
        private static readonly ILogger _eventsLogger =
            LogManager.Instance().LoggerLocator.GetLogger("SystemEvents");

		private SimpleListenerHL7Manager(){}

		///<Developers>
		///	<Developer>Brian Tomlin</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>10/19/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="5808"> 
		///		<ExpectedInput>Valid CPRS HL7Interface parameter and Clinical Order (OMG^O19) message string</ExpectedInput>
		///		<ExpectedOutput>Non-null string representing a valid HL7 acknowledgement message.</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="5809"> 
		///		<ExpectedInput>Null HL7Interface parameter and valid message string</ExpectedInput>
		///		<ExpectedOutput>ArgumentNullException</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="5810"> 
		///		<ExpectedInput>Valid CPRS HL7Interface parameter and null message string</ExpectedInput>
		///		<ExpectedOutput>ArgumentNullException</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="5812"> 
		///		<ExpectedInput>Valid CPRS HL7Interface parameter and unsupported message type of (ADT~A08)</ExpectedInput>
		///		<ExpectedOutput>HL7Exception</ExpectedOutput>
		///	</Case>
		///
		///<Case type="0" testid ="5813"> 
		///		<ExpectedInput>Valid Patient Update HL7Interface parameter and valid message string.</ExpectedInput>
		///		<ExpectedOutput>Null</ExpectedOutput>
		///	</Case>
		///
		///<Case type="0" testid ="5822"> 
		///		<ExpectedInput>Valid Patient Merge HL7Interface parameter and valid message string.</ExpectedInput>
		///		<ExpectedOutput>Null</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// This static method accepts in incoming HL7 message and HL7 interface defined in VBECS and determines 
		/// which HL7 parser should handle the incoming message. All HL7 interfaces with VBECS will need to be 
		/// included in this method as well as the interface name as a constant variable.
		/// CR 2962
		/// </summary>
		/// <param name="hl7Interface"><see cref="HL7Interface"/> defined in the VBECS database with associated </param>
		/// <param name="message">HL7 message string.</param>
		/// <returns>Appropriate HL7 message acknowledgement string or null.</returns>
		public static HL7ProtocolMessage ProcessIncomingHL7Message( HL7Interface hl7Interface, string message )
		{
			if( message == null )
				throw( new ArgumentNullException( "message" ) );

			if( hl7Interface == null ) 
				throw( new ArgumentNullException( "hl7Interface" ) );

			if( hl7Interface.LogEvents )
			{
                // EventLogAppender
                _eventsLogger.Info(string.Format("{0}: {1} received message for {2} interface:\n\n{3}", "Incoming Message", GlobalContext.Instance().ServiceProperties.Name, hl7Interface.InterfaceName, message));
				//EventLog.WriteEntry( "Incoming Message", 
                //    GlobalContext.Instance().ServiceProperties.Name + 
				//	" received message for " + hl7Interface.InterfaceName + " interface:\n\n" + message, EventLogEntryType.Information );
			}

			switch( hl7Interface.InterfaceName )
			{
				// CR 2962
				case BCE_COTS_INTERFACE_NAME:
                    return BceHL7Parser.ParseHL7Message(hl7Interface, message);

				case CPRS_INTERFACE_NAME:
                    return CprsHL7Parser.ParseHL7Message(hl7Interface, message);

				case PATIENT_UPDATE_INTERFACE_NAME:
					return PatientUpdateHL7Parser.ParseHL7Message( hl7Interface, message );

				case PATIENT_MERGE_INTERFACE_NAME:
					return new PatientMergeHL7Parser().ParseHL7Message( hl7Interface, message );

                case AUTO_INSTRUMENT_INTERFACE_NAME:
                    return new AutoInstrumentHL7Parser().ParseHL7Message(hl7Interface, message);

				default:
					return null;
			}
		}


	}
}
